<?php
/**
 * @version    1.0.0
 * @package    com_simplerecipe
 * @author     Simple Recipe <support@simplerecipe.cloud>
 * @copyright  2023 Simple Recipe
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Simplerecipe\Component\Simplerecipe\Administrator\Model;
// No direct access.
defined('_JEXEC') or die;

use \Joomla\CMS\MVC\Model\ListModel;
use \Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use \Joomla\CMS\Factory;
use \Joomla\CMS\Language\Text;
use \Joomla\CMS\Helper\TagsHelper;
use \Joomla\Database\ParameterType;
use \Joomla\Utilities\ArrayHelper;
use Simplerecipe\Component\Simplerecipe\Administrator\Helper\SimplerecipeHelper;

/**
 * Methods supporting a list of Recipes records.
 *
 * @since  1.0.0
 */
class RecipesModel extends ListModel
{
	/**
	* Constructor.
	*
	* @param   array  $config  An optional associative array of configuration settings.
	*
	* @see        JController
	* @since      1.6
	*/
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
				'state', 'a.state',
				'ordering', 'a.ordering',
				'created_by', 'a.created_by',
				'modified_by', 'a.modified_by',
				'title', 'a.title',
				'catid', 'a.catid',
				'parts', 'a.parts',
				'serve_instructions', 'a.serve_instructions',
				'favorite', 'a.favorite',
				'dietary', 'a.dietary',
				'images', 'a.images',
				'hits', 'a.hits',
				'created_time', 'a.created_time',
				'modified_time', 'a.modified_time',
				'from_kitchen', 'a.from_kitchen',
				'serves_number', 'a.serves_number',
			);
		}

		parent::__construct($config);
	}


	

	

	

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string  $ordering   Elements order
	 * @param   string  $direction  Order direction
	 *
	 * @return void
	 *
	 * @throws Exception
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		// List state information.
		parent::populateState('title', 'ASC');

		$context = $this->getUserStateFromRequest($this->context.'.filter.search', 'filter_search');
		$this->setState('filter.search', $context);

		// Split context into component and optional section
		if (!empty($context))
		{
			$parts = FieldsHelper::extract($context);

			if ($parts)
			{
				$this->setState('filter.component', $parts[0]);
				$this->setState('filter.section', $parts[1]);
			}
		}
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * This is necessary because the model is used by the component and
	 * different modules that might need different sets of data or different
	 * ordering requirements.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string A store id.
	 *
	 * @since   1.0.0
	 */
	protected function getStoreId($id = '')
	{
		// Compile the store id.
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.state');

		
		return parent::getStoreId($id);
		
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  DatabaseQuery
	 *
	 * @since   1.0.0
	 */
	protected function getListQuery()
	{
		// Create a new query object.
		$db    = $this->getDbo();
		$query = $db->getQuery(true);

		// Select the required fields from the table.
		$query->select(
			$this->getState(
				'list.select', 'DISTINCT a.*'
			)
		);
		$query->from('`#__simplerecipe_recipes` AS a');
		
		// Join over the users for the checked out user
		$query->select("uc.name AS uEditor");
		$query->join("LEFT", "#__users AS uc ON uc.id=a.checked_out");

		// Join over the user field 'created_by'
		$query->select('`created_by`.name AS `created_by`');
		$query->join('LEFT', '#__users AS `created_by` ON `created_by`.id = a.`created_by`');

		// Join over the user field 'modified_by'
		$query->select('`modified_by`.name AS `modified_by`');
		$query->join('LEFT', '#__users AS `modified_by` ON `modified_by`.id = a.`modified_by`');
		

		// Filter by published state
		$published = $this->getState('filter.state');

		if (is_numeric($published))
		{
			$query->where('a.state = ' . (int) $published);
		}
		elseif (empty($published))
		{
			$query->where('(a.state IN (0, 1))');
		}

		// Filter by search in title
		$search = $this->getState('filter.search');

		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('a.id = ' . (int) substr($search, 3));
			}
			else
			{
				$search = $db->Quote('%' . $db->escape($search, true) . '%');
				$query->where('( a.title LIKE ' . $search . '  OR  a.catid LIKE ' . $search . '  OR  a.favorite LIKE ' . $search . '  OR  a.dietary LIKE ' . $search . ' )');
			}
		}
		

		// Filtering catid
		$filter_catid = $this->state->get("filter.catid");

		if ($filter_catid !== null && !empty($filter_catid))
		{
			$query->where("a.`catid` = '".$db->escape($filter_catid)."'");
		}

		// Filtering dietary
		$filter_dietary = $this->state->get("filter.dietary");

		if ($filter_dietary !== null && (is_numeric($filter_dietary) || !empty($filter_dietary)))
		{
			$query->where('FIND_IN_SET(' . $db->quote($filter_dietary) . ', ' . $db->quoteName('a.dietary') . ')');
		}
		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'title');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		if ($orderCol && $orderDirn)
		{
			$query->order($db->escape($orderCol . ' ' . $orderDirn));
		}

		return $query;
	}

	/**
	 * Get an array of data items
	 *
	 * @return mixed Array of data items on success, false on failure.
	 */
	public function getItems()
	{
		$items = parent::getItems();
		
		foreach ($items as $oneItem)
		{

			if (isset($oneItem->catid))
			{
				$db    = $this->getDbo();
				$query = $db->getQuery(true);

				$query
					->select($db->quoteName('title'))
					->from($db->quoteName('#__categories'))
					->where('FIND_IN_SET(' . $db->quoteName('id') . ', ' . $db->quote($oneItem->catid) . ')');

				$db->setQuery($query);
				$result = $db->loadColumn();

				$oneItem->catid = !empty($result) ? implode(', ', $result) : '';
			}

				// Get the title of every option selected.

				$options = explode(',', $oneItem->dietary);

				$options_text = array();

				foreach ((array) $options as $option)
				{
					$options_text[] = Text::_('COM_SIMPLERECIPE_RECIPES_DIETARY_OPTION_' . strtoupper($option));
				}

				$oneItem->dietary = !empty($options_text) ? implode(',', $options_text) : $oneItem->dietary;
		}

		return $items;
	}
}
